'use strict';

/* --------------------------------------------------------------
 customers_table_controller.js 2020-07-07
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

/**
 * ## Customers Table Controller
 *
 * This controller contains the mapping logic of the customers table.
 *
 * @module Compatibility/customers_table_controller
 */
gx.compatibility.module('customers_table_controller', [gx.source + '/libs/button_dropdown'],

/**  @lends module:Compatibility/customers_table_controller */

function (data) {

    'use strict';

    // ------------------------------------------------------------------------
    // VARIABLES DEFINITION
    // ------------------------------------------------------------------------

    var
    /**
     * Module Selector
     *
     * @var {object}
     */
    $this = $(this),


    /**
     * Default Options
     *
     * @type {object}
     */
    defaults = {
        'adminAccessGranted': false,
        'customerGroupsGranted': false,
        'mailbeezGranted': false,
        'mediafinanzGranted': false,
        'ordersGranted': false
    },


    /**
     * Final Options
     *
     * @var {object}
     */
    options = $.extend(true, {}, defaults, data),


    /**
     * Module Object
     *
     * @type {object}
     */
    module = {},


    /**
     * Reference to the actual file
     *
     * @var {string}
     */
    srcPath = window.location.origin + window.location.pathname,


    /**
     * Query parameter string
     *
     * @type {string}
     */
    queryString = '?' + window.location.search.replace(/\?/, '').replace(/cID=[\d]+/g, '').replace(/action=[\w]+/g, '').concat('&').replace(/&[&]+/g, '&').replace(/^&/g, '');

    // ------------------------------------------------------------------------
    // PRIVATE METHODS
    // ------------------------------------------------------------------------

    /**
     * Map actions for every row in the table.
     *
     * This method will map the actions for each
     * row of the table.
     *
     * @private
     */
    var _mapRowActions = function _mapRowActions() {
        // Iterate over table rows, except the header row
        $('.gx-customer-overview tr').not('.dataTableHeadingRow').each(function () {

            /**
             * Save that "this" scope here
             * @var {object | jQuery}
             */
            var $that = $(this);

            /**
             * Data attributes of current row
             * @var {object}
             */
            var data = $that.data();

            /**
             * Reference to the row action dropdown
             * @var {object | jQuery}
             */
            var $dropdown = $that.find('.js-button-dropdown');

            if ($dropdown.length) {

                // Add click event to the table row and open the
                // customer detail view
                if (data.editable === 1) {
                    $that.find('.btn-edit').closest('tr').css({
                        cursor: 'pointer'
                    }).on('click', function (event) {
                        // Compose the URL and open it
                        var url = [srcPath, '?cID=' + data.rowId, '&action=edit'].join('');
                        if ($(event.target).prop('tagName') === 'TD') {
                            window.open(url, '_self');
                        }
                    });
                }

                // Icon behavior - Edit
                if (data.editable === 1) {
                    $that.find('.btn-edit').css({
                        cursor: 'pointer'
                    }).prop('title', jse.core.lang.translate('edit', 'buttons')).on('click', function () {
                        // Compose the URL and open it
                        var url = [srcPath, '?cID=' + data.rowId, '&action=edit'].join('');
                        window.open(url, '_self');
                    });
                }

                // Icon behavior - Delete
                if (data.rowId !== 1) {
                    $that.find('.btn-delete').css({
                        cursor: 'pointer'
                    }).prop('title', jse.core.lang.translate('delete', 'buttons')).on('click', function () {
                        // Compose the URL and open it
                        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=confirm'].join('');
                        window.open(url, '_self');
                    });
                }

                if (data.rowId === 1) {
                    $that.find('.btn-delete').css({
                        opacity: '0.2'
                    });
                }

                if (data.editable !== 1) {
                    $that.find('.btn-edit').css({
                        opacity: '0.2'
                    });

                    $that.find('.btn-order').css({
                        opacity: '0.2'
                    });
                }

                // Icon behavior - Orders
                if (options.ordersGranted && data.editable === 1) {
                    $that.find('.btn-order').css({
                        cursor: 'pointer'
                    }).prop('title', jse.core.lang.translate('BUTTON_ORDERS', 'admin_buttons')).on('click', function () {
                        // Compose the URL and open it
                        var url = [srcPath.replace('customers.php', 'admin.php'), '?' + $.param({
                            do: 'OrdersOverview',
                            filter: {
                                customer: '#' + data.rowId
                            }
                        })].join('');
                        window.open(url, '_self');
                    });
                }

                _mapButtonDropdown($that, $dropdown, data);
            }
        });
    };

    var _mapButtonDropdown = function _mapButtonDropdown($that, $dropdown, data) {
        var actions = [];

        if (data.editable === 1) {
            actions.push('edit');
            actions.push('BUTTON_EMAIL');
            actions.push('BUTTON_IPLOG');
            actions.push('export_personal_data');
        }

        // Bind dropdown option - Delete
        if (data.rowId !== 1) {
            actions.push('delete');
        }

        if (data.rowId > 1 && data.editable === 1) {
            actions.push('delete_personal_data');
        }

        if ($that.find('[data-cust-group]').data('custGroup') !== 0 && data.editable === 1) {
            actions.push('BUTTON_LOGIN_AS_CUSTOMER');
        }

        if (options.customerGroupsGranted && data.editable === 1) {
            actions.push('BUTTON_STATUS');
        }

        if (options.ordersGranted && data.editable === 1) {
            actions.push('BUTTON_ORDERS');
            actions.push('BUTTON_NEW_ORDER');
        }

        // Admin rights button
        if ($that.find('[data-cust-group]').data('custGroup') === 0 && options.adminAccessGranted && data.editable === 1) {
            actions.push('BUTTON_ACCOUNTING');
        }

        // Bind MailBeez dropdown options.
        var mailBeezConversationsSelector = '.contentTable .infoBoxContent a.context_view_button.btn_right';
        if ($(mailBeezConversationsSelector).length && options.mailbeezGranted && data.editable === 1) {
            actions.push('MAILBEEZ_OVERVIEW');
            actions.push('MAILBEEZ_NOTIFICATIONS');
            actions.push('MAILBEEZ_CONVERSATIONS');
        }

        // Bind Mediafinanz dropdown options.
        var $mediafinanzAction = $('.mediafinanz-creditworthiness');
        if ($mediafinanzAction.length && options.mediafinanzGranted && data.editable === 1) {
            actions.push('BUTTON_MEDIAFINANZ_CREDITWORTHINESS');
        }

        for (var index in actions) {
            _mapCustomerAction($dropdown, actions[index], data);
        }
    };

    var _mapCustomerAction = function _mapCustomerAction($dropdown, action, data) {
        jse.libs.button_dropdown.mapAction($dropdown, action, _sectionMapping[action], function (event) {
            _executeActionCallback(action, data);
        });
    };

    var _sectionMapping = {
        edit: 'buttons',
        delete: 'buttons',
        BUTTON_STATUS: 'admin_buttons',
        BUTTON_ORDERS: 'admin_buttons',
        BUTTON_EMAIL: 'admin_buttons',
        BUTTON_IPLOG: 'admin_buttons',
        MAILBEEZ_OVERVIEW: 'admin_customers',
        MAILBEEZ_NOTIFICATIONS: 'admin_customers',
        MAILBEEZ_CONVERSATIONS: 'admin_customers',
        BUTTON_MEDIAFINANZ_CREDITWORTHINESS: 'admin_buttons',
        BUTTON_NEW_ORDER: 'admin_buttons',
        BUTTON_ACCOUNTING: 'admin_buttons',
        BUTTON_LOGIN_AS_CUSTOMER: 'admin_customers',
        delete_personal_data: 'admin_customers',
        export_personal_data: 'admin_customers'
    };

    /**
     * Get the corresponding callback
     *
     * @param action
     * @private
     */
    var _executeActionCallback = function _executeActionCallback(action, data) {
        switch (action) {
            case 'edit':
                _editCallback(data);
                break;
            case 'delete':
                _deleteCallback(data);
                break;
            case 'BUTTON_STATUS':
                _customerGroupCallBack(data);
                break;
            case 'BUTTON_ORDERS':
                _ordersCallback(data);
                break;
            case 'BUTTON_EMAIL':
                _emailCallback(data);
                break;
            case 'BUTTON_IPLOG':
                _ipLogCallback(data);
                break;
            case 'MAILBEEZ_OVERVIEW':
                _mailBeezOverviewCallback(data);
                break;
            case 'MAILBEEZ_NOTIFICATIONS':
                _mailBeezNotificationsCallback(data);
                break;
            case 'MAILBEEZ_CONVERSATIONS':
                _mailBeezConversationsCallback(data);
                break;
            case 'BUTTON_MEDIAFINANZ_CREDITWORTHINESS':
                _mediafinanzCreditworthinessCallback(data);
                break;
            case 'BUTTON_NEW_ORDER':
                _newOrderCallback(data);
                break;
            case 'BUTTON_ACCOUNTING':
                _adminRightsCallback(data);
                break;
            case 'BUTTON_LOGIN_AS_CUSTOMER':
                _loginAsCustomerCallback(data);
                break;
            case 'delete_personal_data':
                _deletePersonalDataCallback(data);
                break;
            case 'export_personal_data':
                _exportPersonalDataCallback(data);
                break;
            default:
                throw new Error('Callback not found.');
                break;
        }
    };

    var _editCallback = function _editCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, '?cID=' + data.rowId, '&action=edit'].join('');
        window.open(url, '_self');
    };

    var _deleteCallback = function _deleteCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=confirm'].join('');
        window.open(url, '_self');
    };

    var _customerGroupCallBack = function _customerGroupCallBack(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=editstatus'].join('');
        window.open(url, '_self');
    };

    var _ordersCallback = function _ordersCallback(data) {
        // Compose the URL and open it
        var url = [srcPath.replace('customers.php', 'admin.php'), '?' + $.param({
            do: 'OrdersOverview',
            filter: {
                customer: '#' + data.rowId
            }
        })].join('');
        window.open(url, '_self');
    };

    var _emailCallback = function _emailCallback(data) {
        // Compose the URL and open it
        var url = [srcPath.replace('customers.php', 'mail.php'), '?selected_box=tools', '&customer=' + data.custEmail].join('');
        window.open(url, '_self');
    };

    var _ipLogCallback = function _ipLogCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=iplog'].join('');
        window.open(url, '_self');
    };

    var _mailBeezOverviewCallback = function _mailBeezOverviewCallback(data) {
        var $target = $('.contentTable .infoBoxContent a.context_view_button.btn_left');
        var url = $('.contentTable .infoBoxContent a.context_view_button.btn_left').attr('onclick');
        url = url.replace(/cID=(.*)&/, 'cID=' + data.rowId + '&');
        $('.contentTable .infoBoxContent a.context_view_button.btn_left').attr('onclick', url);
        $target.get(0).click();
    };

    var _mailBeezNotificationsCallback = function _mailBeezNotificationsCallback(data) {
        var $target = $('.contentTable .infoBoxContent a.context_view_button.btn_middle');
        var url = $('.contentTable .infoBoxContent a.context_view_button.btn_middle').attr('onclick');
        url = url.replace(/cID=(.*)&/, 'cID=' + data.rowId + '&');
        $('.contentTable .infoBoxContent a.context_view_button.btn_middle').attr('onclick', url);
        $target.get(0).click();
    };

    var _mailBeezConversationsCallback = function _mailBeezConversationsCallback(data) {
        var $target = $('.contentTable .infoBoxContent a.context_view_button.btn_right');
        var url = $('.contentTable .infoBoxContent a.context_view_button.btn_right').attr('onclick');
        url = url.replace(/cID=(.*)&/, 'cID=' + data.rowId + '&');
        $('.contentTable .infoBoxContent a.context_view_button.btn_right').attr('onclick', url);
        $target.get(0).click();
    };

    var _mediafinanzCreditworthinessCallback = function _mediafinanzCreditworthinessCallback(data) {
        var $target = $('.mediafinanz-creditworthiness');
        var onclickAttribute = $target.attr('onclick');
        // Replace the customer number in the onclick attribute. 
        onclickAttribute = onclickAttribute.replace(/cID=(.*', 'popup')/, 'cID=' + data.rowId + '\', \'popup\'');
        $target.attr('onclick', onclickAttribute);
        $target.trigger('click'); // Trigger the click event in the <a> element. 
    };

    var _newOrderCallback = function _newOrderCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, '?cID=' + data.rowId, '&action=new_order'].join('');
        window.open(url, '_self');
    };

    var _adminRightsCallback = function _adminRightsCallback(data) {
        // Compose the URL and open it
        var url = 'admin.php?do=AdminAccess/editAdmin&id=' + data.rowId;
        window.open(url, '_self');
    };

    var _loginAsCustomerCallback = function _loginAsCustomerCallback(data) {
        // Compose the URL and open it
        var url = [srcPath.replace('customers.php', 'admin.php'), '?do=CustomerLogin&customerId=' + data.rowId + '&pageToken=' + jse.core.config.get('pageToken')].join('');
        window.open(url, '_self');
    };

    var _deletePersonalDataCallback = function _deletePersonalDataCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=delete_personal_data'].join('');
        window.open(url, '_self');
    };

    var _exportPersonalDataCallback = function _exportPersonalDataCallback(data) {
        // Compose the URL and open it
        var url = [srcPath, queryString, 'cID=' + data.rowId, '&action=export_personal_data'].join('');
        window.open(url, '_self');
    };

    // ------------------------------------------------------------------------
    // INITIALIZATION
    // ------------------------------------------------------------------------

    module.init = function (done) {
        // Wait until the buttons are converted to dropdown for every row.
        var interval = setInterval(function () {
            if ($('.js-button-dropdown').length) {
                clearInterval(interval);
                _mapRowActions();
            }
        }, 500);

        done();
    };

    return module;
});
//# sourceMappingURL=data:application/json;charset=utf8;base64,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
